function vec4 = multilinear4(odefile,tens4,q1,q2,q3,q4,x0,p,increment)

%--------------------------------------------------------------
%This file computes the multilinear function D(q1,q2,q3,q4) where D =
%D^4(F(x0)), the 4th derivative of the map w.r.t. to phase variables only.
%Directional derivatives are done on unit vectors q1,q2 for accuracy. 
%First we decide whether q1=q2, then the rest. We only compute the
%quantities we need to minimize the amount of work as every function
%evaluation is computationally expensive.
%--------------------------------------------------------------

nphase=size(x0,1);
[nq1,q1]=vecscale(q1);
[nq2,q2]=vecscale(q2);
[nq3,q3]=vecscale(q3);
[nq4,q4]=vecscale(q4);

if (~isempty(tens4)) %Symbolic derivatives if available
  vec4=tensor4op(tens4,q1,q2,q3,q4,nphase);
elseif (isreal(q1) && isreal(q2) && isreal(q3) && isreal(q4))
  if(isequal(q1,q2))
    if(isequal(q1,q3))
      if(isequal(q1,q4))
        vec4 = Dvvvv(odefile,q1,x0,p,increment);
      else
        part1 = Dvvvv(odefile,3.0*q1+q4,x0,p,increment);
        part2 = Dvvvv(odefile,3.0*q1-q4,x0,p,increment);
        part3 = Dvvvv(odefile,q1+q4,x0,p,increment);
        part4 = Dvvvv(odefile,q1-q4,x0,p,increment);
        vec4 = (part1 - part2 - 3.0*part3 + 3.0*part4)/192.0;
      end
    elseif(isequal(q3,q4))
      part1 = Dvvvv(odefile,q1+q3,x0,p,increment);
      part2 = Dvvvv(odefile,q1-q3,x0,p,increment);
      part3 = Dvvvv(odefile,q1,x0,p,increment);
      part4 = Dvvvv(odefile,q3,x0,p,increment);
      vec4 = (part1 + part2 - 2.0*part3 - 2.0*part4)/12.0;
    else
      part1 = Dvvvv(odefile,2.0*q1+q3+q4,x0,p,increment);
      part2 = Dvvvv(odefile,2.0*q1+q3-q4,x0,p,increment);
      part3 = Dvvvv(odefile,2.0*q1-q3+q4,x0,p,increment);
      part4 = Dvvvv(odefile,2.0*q1-q3-q4,x0,p,increment);
      part5 = Dvvvv(odefile, q3+q4,x0,p,increment);
      part6 = Dvvvv(odefile,-q3+q4,x0,p,increment);
      vec4 = (part1 - part2 - part3 + part4 - 2.0*part5 + 2.0*part6)/192.0;
    end
  else
    part1 = Dvvvv(odefile,q1+q2+q3+q4,x0,p,increment);
    part2 = Dvvvv(odefile,q1+q2+q3-q4,x0,p,increment);
    part3 = Dvvvv(odefile,q1+q2-q3+q4,x0,p,increment);
    part4 = Dvvvv(odefile,q1+q2-q3-q4,x0,p,increment);
    part5 = Dvvvv(odefile,q1-q2+q3+q4,x0,p,increment);
    part6 = Dvvvv(odefile,q1-q2+q3-q4,x0,p,increment);
    part7 = Dvvvv(odefile,q1-q2-q3+q4,x0,p,increment);
    part8 = Dvvvv(odefile,q1-q2-q3-q4,x0,p,increment);
    vec4 = (part1 - part2 - part3 + part4 - part5 + part6 + part7 - part8)/192.0;
  end
else %Case of complex vectors q1,q2,q3,q4 call to self with real ones.
  q1r=real(q1);q1i=imag(q1);
  q2r=real(q2);q2i=imag(q2);
  q3r=real(q3);q3i=imag(q3);
  q4r=real(q4);q4i=imag(q4);
  vec4 = multilinear4(odefile,[],q1r,q2r,q3r,q4r,x0,p,increment);
% one imaginary
  vec4 = vec4+(1i)*multilinear4(odefile,[],q1i,q2r,q3r,q4r,x0,p,increment);
  vec4 = vec4+(1i)*multilinear4(odefile,[],q1r,q2i,q3r,q4r,x0,p,increment);
  vec4 = vec4+(1i)*multilinear4(odefile,[],q1r,q2r,q3i,q4r,x0,p,increment);
  vec4 = vec4+(1i)*multilinear4(odefile,[],q1r,q2r,q3r,q4i,x0,p,increment);
% two imaginary
  vec4 = vec4+(-1)*multilinear4(odefile,[],q1r,q2i,q3i,q4r,x0,p,increment);
  vec4 = vec4+(-1)*multilinear4(odefile,[],q1r,q2i,q3r,q4i,x0,p,increment);
  vec4 = vec4+(-1)*multilinear4(odefile,[],q1r,q2r,q3i,q4i,x0,p,increment);
  vec4 = vec4+(-1)*multilinear4(odefile,[],q1i,q2i,q3r,q4r,x0,p,increment);
  vec4 = vec4+(-1)*multilinear4(odefile,[],q1i,q2r,q3i,q4r,x0,p,increment);
  vec4 = vec4+(-1)*multilinear4(odefile,[],q1i,q2r,q3r,q4i,x0,p,increment);
% three imaginary
  vec4 = vec4-(1i)*multilinear4(odefile,[],q1r,q2i,q3i,q4i,x0,p,increment);
  vec4 = vec4-(1i)*multilinear4(odefile,[],q1i,q2r,q3i,q4i,x0,p,increment);
  vec4 = vec4-(1i)*multilinear4(odefile,[],q1i,q2i,q3r,q4i,x0,p,increment);
  vec4 = vec4-(1i)*multilinear4(odefile,[],q1i,q2i,q3i,q4r,x0,p,increment);
  vec4 = vec4+(+1)*multilinear4(odefile,[],q1i,q2i,q3i,q4i,x0,p,increment);
end
vec4=nq1*nq2*nq3*nq4*vec4;

%------------------------------------------------------------
% Computing the fourth order directional derivative w.r.t. vq
function tempvec = Dvvvv(odefile,vq,x0,p,increment)
  f0 = x0;
  f1 = x0 + 4.0*increment*vq;
  f2 = x0 + 2.0*increment*vq;
  f3 = x0 - 2.0*increment*vq;
  f4 = x0 - 4.0*increment*vq;

  f0 = feval(odefile, 0, f0, p{:});
  f1 = feval(odefile, 0, f1, p{:});
  f2 = feval(odefile, 0, f2, p{:});
  f3 = feval(odefile, 0, f3, p{:});
  f4 = feval(odefile, 0, f4, p{:});    

  tempvec = (f1 - 4.0*f2 + 6.0*f0 - 4.0*f3 + f4)/((2*increment)^4);

%----------------------------------------------------
%Scaling q to vector with norm 1, or keep q if smaller to improve accuracy.
function [nq,q_scaled]=vecscale(q)
  if (norm(q)>1)
    nq=norm(q);
    q_scaled=q/nq;
  else
    nq=1;
    q_scaled=q;
  end
