classdef Session < handle
% Main model of the MatCont GUI.
% This object models the 'current state' of the MATLAB GUI. This objects holds the current 'settings',
% the current list of available computation, the current selected computation and the current solution with the current diagram
% GUI-components retrieve information and change the state through this object or one of the submodels.
% 


    properties
        interactiveOutput = 0; % If set to '0', session will remain in command-line mode (non-gui). This is set to 1 by 'GUIMainPanel'


        globalsettings; %contains global settings (CLSettings-object) unrelated to a system: plotoptions and windowpositions

        settings = []; %'current settings' used to start a computation, visualized by starter/continuer/integrator-window
        settingslistener = []; %contains listener-object to the 'settings', this updates everytime the settings-object is swapped

        computation = []; %current computation (ContConf). The computation defines the content of 'settings'
        
        solutionhandle = []; %contains a handle with the current diagram and the current solution (if any).
        
        branchmanager; %submodel that maintains the list of available computations for a given state
        windowmanager; %submodels that keeps tracks of windowhandles and positions.
        outputmanager; %submodel that maintains the list of current outputs: plot2d, plot3d and numeric.
        
        systemspath = ''; %absolute path to the 'Systems/' map

        pointloader; %pointloader is passed along to plots to enable the selecting of initial points through clicking on pointlabels
        
        locked = 0; %locked keeps track if a computation is in progress. This prevents modification during computation
        
        
    end
    
    properties(Constant)
       VERSION = 0.75; %session will not load in states generated by a session with a lower version number. If major changes happen to the structure, make this number higher.
    end
    
    
    events
        settingChanged %changing of a setting in the settings in the 'setting' field
        settingsChanged %reconfiguration of 'settings' or replacing of 'settings'
        computationChanged %selecting of a new computation (Current Curve in mainwindow)
        solutionChanged %changing of the current solution/diagram
        diagramChanged %changing of the current diagram (but not solution)
        initpointChanged %new initial point-type is set in the current settings
        lockChanged %changing of the lock-status (computing/not-computing)
        shutdown %activated when the GUI shuts down
    end
    
    methods
        function obj = Session()
            %Creates an inital empty clean session.

            obj.locked = 0;
            %path to 'Systems' map and store it as an anchor
            obj.systemspath = getSystemsDir();
            %empty Computation configuration
            obj.computation = CompConf();
            %empty output manager
            obj.outputmanager = GUIOutputManager(obj);
            %empty settings
            settings = CLSettings();
            %place empty system
            settings.installSetting('system', CLSystem());
            obj.changeSettings(settings);
            %empty solution + diagram
            obj.solutionhandle = CLSolutionHandler();
            %empty branchmanager
            obj.branchmanager = CLBranchManager(obj);
            %empty pointloader
            obj.pointloader = CLPointLoader(obj);
            %construction of new 'global' settings, initialized by CLglobalsettings
            obj.globalsettings = CLglobalsettings(obj);
            %empty windowmanager
            obj.windowmanager = GUIWindowManager(obj);
            
        end
        
        function p = getSystemsPath(obj)
            %returns absolut path to '/Systems/'
            p = obj.systemspath;
        end
        
        
        function onlyChangeSystem(obj, odesys)
            %re-initialize this session object for a new system with an empty state.
            obj.computation = CompConf();  %empty
            settings = CLSettings(); %empty
            settings.installSetting('system', odesys); %place system in settings
            obj.solutionhandle = CLSolutionHandler.fromSystem(odesys); %create empty 'solution' + select default diagram
            obj.changeSettings(settings); 
            obj.pointloader.clear(); 
            obj.notify('computationChanged');
        end
        
        function changeSettings(obj, settings)
            %changes the 'settings' objects.
            obj.settings = settings;
            %check if parameters/coordinates/etc still match the system
            obj.settings.sanityCheck();
            %make all existing settings invisible to the user
            obj.settings.softClear();
            if ~isempty(obj.computation)
                %let the computation object configure the settings, place default values if new setting-objects are introduced in settings
                obj.computation.configureSettings(obj.settings);
            end
            obj.settings.compconf = obj.computation;
            
            obj.settings.versionnumber = obj.VERSION; %settings with older version numbers can be blocked
            obj.notify('settingsChanged');
            obj.notify('initpointChanged');

            % delete listener for old settings, subscribe to new settings
            delete(obj.settingslistener);
            obj.settingslistener = obj.settings.addlistener('settingChanged', @(o, e) obj.notify('settingChanged'));
            clear 'settings';
            global settings; settings = obj.settings;  %export to global variable as a command-line interface
        end

        function setInteractiveOutput(obj, b)
            obj.interactiveOutput = b;
        end
        function wm = getWindowManager(obj)
            wm = obj.windowmanager;
        end
        function om = getOutputManager(obj)
            om = obj.outputmanager;
        end
        function changeInitPoint(obj, point)
            % See Menu-item
            obj.settings.IP.set(point);
            obj.notify('initpointChanged');
            
        end
        
        function cfgs = configs(obj)
            %obtain a list of available computations based on the current state
            cfgs = obj.branchmanager.getCompConfigs();
        end
        
        
        function selectCompConf(obj, compconf)
            % computation (compconf) selected
            obj.computation = compconf;
            %make the settings invisible
            obj.settings.softClear();
            %let the new computation configure the settings
            compconf.configureSettings(obj.settings);

            obj.notify('settingsChanged');
            obj.notify('computationChanged');
        end
        
        function disp(obj)
            %displays session in Command Line Window
            comp = '<none>';
            if ~isempty(obj.computation)
                comp = obj.computation.toString();
            end
            fprintf('computation: %s\n', comp);
            disp(obj.settings); 
            fprintf([repmat('-', 1, 25) '\n']);
            name = obj.retrieveCLName();  %obtain the name of session in 'base'/command-line
            %print shortcuts to functions:
            fprintf('<a href="matlab: %s.select()">View Computations</a>, ', name);
            fprintf('<a href="matlab: %s.compute()">View Actions</a>, ', name);
            fprintf('<a href="matlab: %s.switches()">View Switches</a>.\n', name);
        end
        
        
        
        function select(obj, index)
            % for use in Command-Line
            % select a computation from the list of available computations, no argument lists the computations
            cfgs = obj.configs();
            if nargin < 2
                name = obj.retrieveCLName(); %get name of 'session' in 'base'-namespace
                %list available computations:
                for index = 1:length(cfgs)
                    cfg = cfgs{index};
                    fprintf('<a href="matlab: %s.select(%i)">[%2i]</a> %s\n',name, index, index, cfg.toStringCL());
                end
            else
                %select a computation from the list
                obj.selectCompConf(cfgs{index});
            end
        end
        
        
        function computeSolution(obj, action)
            % This function instructs the session to use to current computation to produce a new solution using the current settings.
            % 'action' is a struct produced by the current computation-object.

            % an action can block the execution based on the current settings/solution
            if ~isempty(action.function) && action.valid(obj.settings, obj.solutionhandle.solution)
                
                %lock down GUI to prevent input, preparing for computation
                obj.lock(); drawnow
                try
                    %produce new solution based on action (obtain in a computation-object)
                    [solution, errmsg, overwrite] = action.function(obj.settings, obj.solutionhandle.solution);
                    
                    %empty solution means error
                    if isempty(solution)
                        obj.windowmanager.closeWindow('monitor'); drawnow;
                        if obj.interactiveOutput
                            if ~isempty(errmsg)
                                errordlg(errmsg , 'Computation can not be started');
                            end
                        else
                            fprintf(2, sprintf('Failure: %s\n', errmsg));
                        end
                        
                    else  %computation was succesfully produced
                        %if overwrite is set to 'true', the new solution replaced the old (overwritten)
                        %if set to 'false', the new solution is added.
                        if overwrite
                            %re-use path
                            path = obj.solutionhandle.solutionpath;
                        else
                            %create new path/filename
                            path = obj.solutionhandle.getDefaultName(obj);
                        end
                        
                   
                        %ma rimuove colonne...
                        
                        %write solution to file
                        %{ 
                        qui
                        if(obj.settings.fields.system.sys_type=="DDE")
                            %per ogni coord qui
                            salva=obj.settings.fields.system.coordinates(1,:);
                            for i=1:obj.settings.fields.system.dim
                                for j=1:obj.settings.fields.system.no_discretizationPoints
                                    current=salva{i}+"_"+j;
                                    obj.settings.fields.system.coordinates{end+1}=current;
                                end
                            end
                            obj.settings.fields.system.dim=obj.settings.fields.system.no_discretizationPoints*obj.settings.fields.system.dim+1;

                        end
                        %}
                        obj.solutionhandle.store(path, solution);
                        obj.pointloader.assignPath(path); %update references in pointloader, preceded by 'newLabel'
                    end
                    
                    
                catch error %uncontrolled crash
                    obj.windowmanager.closeWindow('monitor'); drawnow;
                    errordlg(error.message , 'An unexpected error occured (see Command Window)');
		    message = regexp(getReport(error), '\n', 'split');
		    fprintf(message{2});fprintf('\n\n');   
                end
                %unlock GUI, return to normal
                obj.unlock();
                obj.notify('solutionChanged');
                %export solution to global var (CLI)
                clear 'solution'; global solution; solution = obj.solutionhandle.solution; 

            end
        end
        
        function alist = getComputeActions(obj)
            % get a list of 'actions' from a computation. These will often be 'forward', 'backward' and 'extend'
            if ~isempty(obj.computation)
                alist = obj.computation.actions();
            else
                alist = [];
            end
        end
        
        function monitor(obj)
            % set up the 'Monitor' window and prepare the GUI for interacting with the continuer.


            global sOutput;  %The global var 'soutput' (SessionOutput) is used by the continuer to communicate with GUI
            sOutput = SessionOutput(obj); %create new;
            %have the output manager preparing numeric/plot windows and setting them up to receive continuer output
            obj.outputmanager.configureSessionOutput();
            %start the monitor window (pause/resume/stop)
            GUIMonitorPanel(obj, obj.windowmanager);
           
        end
        
        function compute(obj, index)
            %For use in Command-Window. Get a list of 'actions' from a computation (if no arguments).  Execute an action if argument is given. (Actions: forward/backward/extend)
            actions = obj.computation.actions();  %list with actions
            if nargin < 2
                %list the actions in the Command-Window
                name = obj.retrieveCLName(); %name of 'session' in command-line
                for index = 1:length(actions)
                    if ~isempty(actions(index).function)
                        label = actions(index).label;
                        fprintf('<a href="matlab: %s.compute(%i)">[%2i]</a> %s\n',name, index, index, label);
                    end
                end
            else
                %execute an action (by index)

                %create a reference to a new curve in 'pointloader' for the solution that is about to be computed
                obj.pointloader.newLabel();
                
                if obj.interactiveOutput
                    obj.monitor();  %start up GUI-link with continuer
                end
                obj.computeSolution(actions(index));
                if  ~obj.interactiveOutput  %if in command-line mode: display the solution
                    disp(obj.solutionhandle.solution);
                end
                
            end
        end

        function changeSystem(obj, system)
           %only change system (shortcut)
           obj.changeState(system, [], [], []); 
        end
        
        function changeState(obj, system, diagram, solution, newsettings)
            % Change System/diagram/solution/newsettings in Session changing the current state.
            % arguments:
            %   system, [], [], [] -> change only system
            %   system, diagram, [], [] -> change system and diagram (or only diagram if system does not differ)
            %   system, diagram, solution, [] -> change solution,  diagram and system are changed if they differ from the current state
            %   system, diagram, solution, newsettings -> change settings, change others if they differ from current state. 
            %  
            % 'newsettings' is often generated by a solution based on a special point.
            %
            %
            oldsysname = obj.getSystemName();

            
            obj.saveToFile(@(s) 1, obj.outputmanager.savePlotConfs(), 'local');
            
            
            if isempty(diagram) && isempty(solution) && isempty(newsettings)
                obj.onlyChangeSystem(system);
                
            elseif isempty(solution) && isempty(newsettings)
                %fprintf('change diagram\n');
                if ~obj.hasSystem() || ~strcmp(obj.getSystem().getName(), system.getName())
                    obj.onlyChangeSystem(system);
                end
                obj.solutionhandle = CLSolutionHandler.fromDiagram(diagram);
                obj.notify('solutionChanged');
                
            elseif isempty(newsettings)
                obj.solutionhandle = CLSolutionHandler.fromSolution(diagram, solution);

                obj.computation = solution.compbranch;
                newsettings = solution.settings.copy();
                newsettings.installSetting('system', system);

                obj.branchmanager.sync(obj, obj.computation, newsettings);
                obj.changeSettings(newsettings);
                
                %obj.notify('settingsChanged'); -> done by changeSetting
                %obj.notify('initpointChanged'); -> done by changeSetting
                obj.notify('computationChanged');
                obj.notify('solutionChanged');
            else
                %system changed by newsettings.
                obj.solutionhandle = CLSolutionHandler.fromDiagram(diagram);
                newsettings.installSetting('system', system);
                obj.changeSettings(newsettings);
                obj.notify('solutionChanged');
            end
            
            if ~strcmp(oldsysname, obj.getSystemName())
                
                obj.outputmanager.onChangeSystem(); %close plot windows, etc.
            end
            clear 'solution'; global solution; solution = obj.solutionhandle.solution;  %export to global variables.
            
        end
        
        function saveToFile(obj, windowrestore, plotconfs, ~) %fourth: local
            try
                if nargin >= 4 
                    odesys = obj.settings.system;  
                    if isempty(odesys); return; end
                    if ~exist(fullfile(obj.systemspath, odesys.getName()), 'dir')
                        mkdir(fullfile(obj.systemspath, odesys.getName()));
                    end
                    filename = fullfile(obj.systemspath, odesys.getName(), 'session.mat');
                else
                    filename = fullfile(obj.systemspath, 'session.mat');
                end
                settings = obj.settings;
                computation = obj.computation;
                globalsettings = obj.globalsettings;
                solutionhandle = obj.solutionhandle;

                save(filename, 'settings', 'computation', 'globalsettings', 'solutionhandle', 'windowrestore', 'plotconfs');
            catch error
                  errordlg(error.message , 'An unexpected error occured (see Command Window)');
                  fprintf(2, getReport(error)); fprintf('\n\n')
               
            end
        end
        %qui carica da file
        function [windowrestore, plotconfs] = loadFromFile(obj, ~) %second: local
            windowrestore = @(s) []; plotconfs = {};
            try
                odesys = [];
                if nargin >= 2
                    odesys = obj.settings.system; 
                    if isempty(odesys); return; end
                    filename = fullfile(obj.systemspath, odesys.getName(), 'session.mat');
                else
                    filename = fullfile(obj.systemspath, 'session.mat');
                end
                o = load(filename);
                
                if obj.VERSION ~= o.settings.versionnumber
                    fprintf(2, 'warning: rejecting loading in a state from an older version of MatCont: %s\n\n', filename);
                   return 
                end
                
                obj.settings = o.settings;
                odesys_new = o.settings.system;
                odesys_new = CLSystem(fullfile(obj.systemspath,[odesys_new.name '.mat']));
                if isempty(odesys_new.handle)
                    odesys_new.handle = odesys.handle;
                end
                assert(~isempty(odesys_new.handle));
                obj.settings.installSetting('system', odesys_new);
                global settings; settings = obj.settings;
                obj.computation = o.computation;
                if obj.VERSION == o.globalsettings.versionnumber
                    obj.globalsettings = o.globalsettings;
                end
                obj.solutionhandle = o.solutionhandle;  
                obj.solutionhandle.syncHandleWithMachine();
                obj.branchmanager.sync(obj, obj.computation, obj.settings)
                obj.windowmanager = GUIWindowManager(obj);
                obj.notify('settingsChanged');
                windowrestore = o.windowrestore;
                plotconfs = o.plotconfs;
                
            catch
            end
            
        end
        
        function shutdownGUI(obj)
            windowrestore = obj.windowmanager.restoreFunction();
            plotconfs = obj.outputmanager.savePlotConfs();
            obj.outputmanager.onChangeSystem(); %close plot windows.
            obj.windowmanager.closeSubWindows();
            obj.saveToFile(windowrestore, plotconfs);
            obj.notify('shutdown');
            
        end
        
        function switches(obj, index)
            switchlist = [];
            if ~isempty(obj.solutionhandle.solution)
                switchlist = obj.solutionhandle.solution.listSwitches();
                
            end
            if nargin < 2
                name = obj.retrieveCLName();
                for index = 1:length(switchlist)
                    fprintf('<a href="matlab: %s.switches(%i)">[%2i]</a> %s\n',name, index, index, switchlist.toString(index));
                    
                end
            else
                newsettings = switchlist.activateSwitch(index);
                if ~isempty(newsettings)
                    obj.changeSettings(newsettings);
                    disp(obj);
                end
            end
            
            
        end
        
        
        
        function name = retrieveCLName(obj)
            result = evalin('base', 'whos');
            search = strcmp({result.class}, 'Session');
            if sum(search) ~= 1
                name = [];
            else
                name = result(find(search)).name;
            end
        end
        % % %
        % %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%5
        function sys = getSystem(obj)
            sys = obj.settings.system;
        end
        
        function name = getSystemName(obj)
           sys = obj.getSystem();
           if isempty(sys)
               name = '';
           else
              name = sys.getName(); 
           end
            
            
        end
        
        function b = hasSystem(obj)
            b = ~isempty(obj.settings.system);
        end
        function o = getCompConf(obj)
            o = obj.computation;
        end
        function IP = getPointType(obj)
            IP = obj.settings.IP;
        end
        function OI = getOutputInterpreter(obj)
            OI = obj.computation.getOutputInterpreter();
            
        end
        function b = hasSolution(obj)
            b = ~isempty(obj.solutionhandle.solution);
            
        end
        function sh = getSolutionHandle(obj)
            sh = obj.solutionhandle;
        end
        function o = getSolution(obj)
            o = obj.solutionhandle.solution;
            o.name = obj.solutionhandle.getSolutionName();
            
        end
        function b = hasDiagram(obj)
            b = ~isempty(obj.solutionhandle.diagram);
        end
        
        function d = getDiagram(obj)
            d = obj.solutionhandle.diagram;
            
        end
        function o = solution(obj)
            fprintf(2, 'deprecated ...\n');
            o = obj.getSolution();
        end
        
        % % % % % LOCK % % %
        function b = isUnlocked(obj); b = ~obj.locked; end
        function b = isLocked(obj); b = obj.locked; end
        function lock(obj)
            obj.locked = 1;
            obj.notify('lockChanged');
        end
        function unlock(obj)
            obj.locked = 0;
            obj.notify('lockChanged');
        end
        
        function resetState(obj)
           obj.unlock(); 
           figs = allchild(0);
           for k = 1:length(figs)
                fig = figs(k);
                if strcmpi(fig.Visible, 'off') && strcmpi(fig.Tag, 'matcont')
                   delete(fig); 
                end
               
           end
           
           eventnames = events(obj);
           for k = 1:length(eventnames)
              if ~strcmp('ObjectBeingDestroyed', eventnames{k})
                obj.notify(eventnames{k}); 
              end
           end
           
           global session; session = obj; %place reference to session in global variable.
           
        end
        
    end
end
